<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WC_WhatsApp_Waappir_Shortcode_Processor {

    /**
     * Process shortcodes in message for an order.
     *
     * @param string   $message
     * @param WC_Order $order
     *
     * @return string
     */
    public static function process_message( $message, $order ) {
        if ( ! $order instanceof WC_Order ) {
            return $message;
        }

        // Order details
        $billing_phone = $order->get_billing_phone();
        $formatted_phone = WC_WhatsApp_Waappir_Messenger::format_phone_number( $billing_phone );

        $replacements = array(
            '{mobile}'           => $formatted_phone,
            '{phone}'            => $formatted_phone,
            '{email}'            => $order->get_billing_email(),
            '{status}'           => wc_get_order_status_name( $order->get_status() ),
            '{all_items}'        => self::get_order_items_list( $order, 'simple' ),
            '{all_items_full}'   => self::get_order_items_list( $order, 'full' ),
            '{all_items_qty}'    => self::get_order_items_list( $order, 'with_qty' ),
            '{count_items}'      => count( $order->get_items() ),
            '{price}'            => $order->get_total(),
            '{post_id}'          => $order->get_id(),
            '{order_id}'         => $order->get_order_number(),
            '{transaction_id}'   => $order->get_transaction_id(),
            '{date}'             => $order->get_date_created() ? $order->get_date_created()->date_i18n() : '',
            '{description}'      => $order->get_customer_note(),
            '{payment_method}'   => $order->get_payment_method_title(),
            '{shipping_method}'  => $order->get_shipping_method(),
        );

        // Billing details
        $replacements['{b_first_name}'] = $order->get_billing_first_name();
        $replacements['{b_last_name}']  = $order->get_billing_last_name();
        $replacements['{b_company}']     = $order->get_billing_company();
        $replacements['{b_country}']     = $order->get_billing_country();
        $replacements['{b_state}']       = $order->get_billing_state();
        $replacements['{b_city}']        = $order->get_billing_city();
        $replacements['{b_address_1}']   = $order->get_billing_address_1();
        $replacements['{b_address_2}']   = $order->get_billing_address_2();
        $replacements['{b_postcode}']    = $order->get_billing_postcode();

        // Shipping details
        $replacements['{sh_first_name}'] = $order->get_shipping_first_name();
        $replacements['{sh_last_name}']  = $order->get_shipping_last_name();
        $replacements['{sh_company}']     = $order->get_shipping_company();
        $replacements['{sh_country}']     = $order->get_shipping_country();
        $replacements['{sh_state}']       = $order->get_shipping_state();
        $replacements['{sh_city}']        = $order->get_shipping_city();
        $replacements['{sh_address_1}']   = $order->get_shipping_address_1();
        $replacements['{sh_address_2}']   = $order->get_shipping_address_2();
        $replacements['{sh_postcode}']    = $order->get_shipping_postcode();

        // Tracking codes (if available via custom fields)
        $replacements['{post_tracking_code}'] = $order->get_meta( '_post_tracking_code', true );
        $replacements['{post_tracking_url}']  = $order->get_meta( '_post_tracking_url', true );

        // Replace all shortcodes
        return str_replace( array_keys( $replacements ), array_values( $replacements ), $message );
    }

    /**
     * Get order items list in different formats.
     *
     * @param WC_Order $order
     * @param string   $format
     *
     * @return string
     */
    protected static function get_order_items_list( $order, $format = 'simple' ) {
        $items = array();

        foreach ( $order->get_items() as $item ) {
            $product = $item->get_product();
            if ( ! $product ) {
                continue;
            }

            switch ( $format ) {
                case 'full':
                    // Include variation attributes
                    $item_name = $item->get_name();
                    if ( $product->is_type( 'variation' ) ) {
                        $item_name .= ' - ' . wc_get_formatted_variation( $product, true );
                    }
                    $items[] = $item_name;
                    break;

                case 'with_qty':
                    $items[] = $product->get_name() . ' (×' . $item->get_quantity() . ')';
                    break;

                case 'simple':
                default:
                    $items[] = $product->get_name();
                    break;
            }
        }

        return implode( ', ', $items );
    }

    /**
     * Process stock shortcodes for low/out of stock messages.
     *
     * @param string     $message
     * @param WC_Product $product
     *
     * @return string
     */
    public static function process_stock_message( $message, $product ) {
        if ( ! $product instanceof WC_Product ) {
            return $message;
        }

        $replacements = array(
            '{product_name}'  => $product->get_name(),
            '{product_id}'    => $product->get_id(),
            '{product_sku}'   => $product->get_sku(),
            '{stock_qty}'     => $product->get_stock_quantity(),
            '{product_price}' => $product->get_price(),
        );

        return str_replace( array_keys( $replacements ), array_values( $replacements ), $message );
    }
}