<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WC_WhatsApp_Waappir_Order_Handler {

    /**
     * Initialize order handler hooks.
     */
    public static function init() {
        // Hook into order status changes
        add_action( 'woocommerce_order_status_changed', array( __CLASS__, 'handle_order_status_change' ), 10, 4 );

        // Hook into new order creation
        add_action( 'woocommerce_new_order', array( __CLASS__, 'handle_new_order' ), 10, 1 );
    }

    /**
     * Handle order status change.
     *
     * @param int    $order_id
     * @param string $old_status
     * @param string $new_status
     * @param WC_Order $order
     */
    public static function handle_order_status_change( $order_id, $old_status, $new_status, $order ) {
        // Check if WhatsApp service is enabled
        if ( 'yes' !== get_option( 'wc_whatsapp_waappir_enabled', 'no' ) ) {
            return;
        }

        // Send message to admin if enabled
        self::send_admin_notification( $order, $new_status );

        // Send message to customer if enabled
        self::send_customer_notification( $order, $new_status );
    }

    /**
     * Send notification to admin.
     *
     * @param WC_Order $order
     * @param string   $new_status
     */
    protected static function send_admin_notification( $order, $new_status ) {
        // Check if admin notifications are enabled
        if ( 'yes' !== get_option( 'wc_whatsapp_waappir_admin_enabled', 'no' ) ) {
            return;
        }

        // Check if this status should trigger notification
        $selected_statuses = get_option( 'wc_whatsapp_waappir_admin_statuses', array() );
        if ( ! in_array( $new_status, $selected_statuses, true ) ) {
            return;
        }

        // Get admin phone
        $admin_phone = get_option( 'wc_whatsapp_waappir_admin_phone', '' );
        if ( empty( $admin_phone ) ) {
            return;
        }

        // Get message template
        $msg_key = 'wc_whatsapp_waappir_admin_msg_' . $new_status;
        $message_template = get_option( $msg_key, '' );
        if ( empty( $message_template ) ) {
            return;
        }

        // Process shortcodes
        $message = WC_WhatsApp_Waappir_Shortcode_Processor::process_message( $message_template, $order );

        // Send message
        WC_WhatsApp_Waappir_Messenger::send_message_via_api( $admin_phone, $message );
    }

    /**
     * Handle new order creation.
     *
     * @param int $order_id
     */
    public static function handle_new_order( $order_id ) {
        // Check if WhatsApp service is enabled
        if ( 'yes' !== get_option( 'wc_whatsapp_waappir_enabled', 'no' ) ) {
            return;
        }

        // Check if customer notifications are enabled
        if ( 'yes' !== get_option( 'wc_whatsapp_waappir_customer_enabled', 'no' ) ) {
            return;
        }

        // Check if new order notifications are enabled
        if ( 'yes' !== get_option( 'wc_whatsapp_waappir_customer_new_order_enabled', 'no' ) ) {
            return;
        }

        $order = wc_get_order( $order_id );
        if ( ! $order ) {
            return;
        }

        // Get customer phone
        $customer_phone = $order->get_billing_phone();
        if ( empty( $customer_phone ) ) {
            return;
        }

        // Get new order message template
        $message_template = get_option( 'wc_whatsapp_waappir_customer_new_order_msg', '' );
        if ( empty( $message_template ) ) {
            // Default message if template is empty
            $message_template = 'سفارش شما با شماره {order_id} ثبت شد. با تشکر از خرید شما!';
        }

        // Process shortcodes
        $message = WC_WhatsApp_Waappir_Shortcode_Processor::process_message( $message_template, $order );

        // Send message
        WC_WhatsApp_Waappir_Messenger::send_message_via_api( $customer_phone, $message );
    }

    /**
     * Send notification to customer.
     *
     * @param WC_Order $order
     * @param string   $new_status
     */
    protected static function send_customer_notification( $order, $new_status ) {
        // Check if customer notifications are enabled
        if ( 'yes' !== get_option( 'wc_whatsapp_waappir_customer_enabled', 'no' ) ) {
            return;
        }

        // Check if this status should trigger notification
        $selected_statuses = get_option( 'wc_whatsapp_waappir_customer_statuses', array() );
        if ( ! in_array( $new_status, $selected_statuses, true ) ) {
            return;
        }

        // Get customer phone
        $customer_phone = $order->get_billing_phone();
        if ( empty( $customer_phone ) ) {
            return;
        }

        // Get message template
        $msg_key = 'wc_whatsapp_waappir_customer_msg_' . $new_status;
        $message_template = get_option( $msg_key, '' );
        if ( empty( $message_template ) ) {
            return;
        }

        // Process shortcodes
        $message = WC_WhatsApp_Waappir_Shortcode_Processor::process_message( $message_template, $order );

        // Send message
        WC_WhatsApp_Waappir_Messenger::send_message_via_api( $customer_phone, $message );
    }
}