<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WC_WhatsApp_Waappir_Messenger {

    const RATE_LIMIT_TRANSIENT = 'wc_whatsapp_waappir_last_test_sent';
    const RATE_LIMIT_SECONDS   = 60;

    /**
     * Format phone number for WhatsApp API.
     * Converts Iranian phone numbers starting with '09' to '989'.
     *
     * @param string $phone
     *
     * @return string
     */
    public static function format_phone_number( $phone ) {
        // Remove all non-numeric characters
        $phone = preg_replace( '/\D/', '', $phone );

        // If phone starts with '09' (Iranian mobile), convert to '989'
        if ( preg_match( '/^09\d{9}$/', $phone ) ) {
            $phone = '989' . substr( $phone, 2 );
        }

        return $phone;
    }

    /**
     * Send test message respecting rate limiting.
     *
     * @param string $phone
     * @param string $message
     *
     * @return true|WP_Error
     */
    public static function send_test_message( $phone, $message ) {
        $enabled = get_option( 'wc_whatsapp_waappir_enabled', 'no' );

        if ( 'yes' !== $enabled ) {
            return new WP_Error( 'wc_whatsapp_waappir_disabled', __( 'وب‌سرویس واتساپ غیرفعال است. ابتدا آن را در تب وبسرویس فعال کنید.', 'wc-whatsapp-waappir' ) );
        }

        if ( empty( $phone ) || empty( $message ) ) {
            return new WP_Error( 'wc_whatsapp_waappir_missing_fields', __( 'شماره و پیام باید وارد شوند.', 'wc-whatsapp-waappir' ) );
        }

        // Rate limiting.
        $last_sent = get_transient( self::RATE_LIMIT_TRANSIENT );
        if ( $last_sent ) {
            $remaining = self::RATE_LIMIT_SECONDS - ( time() - (int) $last_sent );
            if ( $remaining > 0 ) {
                /* translators: %d: seconds */
                return new WP_Error( 'wc_whatsapp_waappir_rate_limited', sprintf( __( 'لطفاً %d ثانیه دیگر برای ارسال پیام تست صبر کنید.', 'wc-whatsapp-waappir' ), $remaining ) );
            }
        }

        $result = self::send_message_via_api( $phone, $message );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        // Set rate limit.
        set_transient( self::RATE_LIMIT_TRANSIENT, time(), self::RATE_LIMIT_SECONDS );

        return true;
    }

    /**
     * Send message via configured REST API.
     *
     * @param string $phone
     * @param string $message
     *
     * @return array|WP_Error
     */
    public static function send_message_via_api( $phone, $message ) {
        $api_url    = get_option( 'wc_whatsapp_waappir_api_url', 'https://panel.waapp.ir/api/v1/send' );
        $auth_token = get_option( 'wc_whatsapp_waappir_auth_token', '' );

        if ( empty( $api_url ) ) {
            return new WP_Error( 'wc_whatsapp_waappir_missing_api_url', __( 'آدرس وب‌سرویس تنظیم نشده است.', 'wc-whatsapp-waappir' ) );
        }

        if ( empty( $auth_token ) ) {
            return new WP_Error( 'wc_whatsapp_waappir_missing_token', __( 'توکن Authorization تنظیم نشده است.', 'wc-whatsapp-waappir' ) );
        }

        // Format phone number for WhatsApp
        $formatted_phone = self::format_phone_number( $phone );

        $body = array(
            'phone'   => $formatted_phone,
            'message' => $message,
        );

        $args = array(
            'method'      => 'POST',
            'headers'     => array(
                'Authorization' => 'Bearer ' . $auth_token,
                'Content-Type'  => 'application/json',
            ),
            'body'        => wp_json_encode( $body ),
            'timeout'     => 15,
            'data_format' => 'body',
        );

        $response = wp_remote_post( $api_url, $args );

        if ( is_wp_error( $response ) ) {
            return new WP_Error( 'wc_whatsapp_waappir_http_error', sprintf( __( 'خطا در اتصال به وب‌سرویس: %s', 'wc-whatsapp-waappir' ), $response->get_error_message() ) );
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );

        if ( $code < 200 || $code >= 300 ) {
            return new WP_Error( 'wc_whatsapp_waappir_api_error', sprintf( __( 'وب‌سرویس وضعیت ناموفق برگرداند (کد %1$d): %2$s', 'wc-whatsapp-waappir' ), $code, $body ) );
        }

        return $response;
    }
}
