<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WC_WhatsApp_Waappir_Admin_Page {

    const PAGE_SLUG = 'wc-whatsapp-waappir-settings';

    /**
     * Render admin page.
     */
    public static function render_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'شما دسترسی لازم برای مشاهده این صفحه را ندارید.', 'wc-whatsapp-waappir' ) );
        }

        $active_tab = isset( $_GET['tab'] ) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : 'webservice'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

        // Handle form submissions.
        self::handle_post( $active_tab );

        echo '<div class="wrap">';
        echo '<h1>' . esc_html__( 'پیکربندی واتساپ وااپ', 'wc-whatsapp-waappir' ) . '</h1>';

        self::render_tabs( $active_tab );

        echo '<div class="wc-whatsapp-waappir-tab-content">';

        switch ( $active_tab ) {
            case 'admin_message':
                self::render_admin_message_tab();
                break;

            case 'customer_message':
                self::render_customer_message_tab();
                break;

            case 'features':
                self::render_features_tab();
                break;

            case 'about':
                self::render_about_tab();
                break;

            case 'webservice':
            default:
                self::render_webservice_tab();
                break;
        }

        echo '</div>'; // .wc-whatsapp-waappir-tab-content
        echo '</div>'; // .wrap
    }

    /**
     * Render tabs navigation.
     *
     * @param string $active_tab
     */
    protected static function render_tabs( $active_tab ) {
        $tabs = array(
            'webservice'    => __( 'وبسرویس', 'wc-whatsapp-waappir' ),
            'admin_message' => __( 'پیام به مدیر کل', 'wc-whatsapp-waappir' ),
            'customer_message' => __( 'پیام به مشتری', 'wc-whatsapp-waappir' ),
            'features'      => __( 'ویژگی‌های قدرتمند', 'wc-whatsapp-waappir' ),
            'about'         => __( 'درباره ما', 'wc-whatsapp-waappir' ),
        );

        echo '<h2 class="nav-tab-wrapper">';

        foreach ( $tabs as $id => $label ) {
            $class = ( $active_tab === $id ) ? ' nav-tab-active' : '';
            $url   = add_query_arg(
                array(
                    'page' => self::PAGE_SLUG,
                    'tab'  => $id,
                ),
                admin_url( 'admin.php' )
            );

            printf(
                '<a href="%1$s" class="nav-tab%3$s">%2$s</a>',
                esc_url( $url ),
                esc_html( $label ),
                esc_attr( $class )
            );
        }

        echo '</h2>';
    }

    /**
     * Handle POST submissions for different tabs.
     *
     * @param string $active_tab
     */
    protected static function handle_post( $active_tab ) {
        if ( 'webservice' === $active_tab && isset( $_POST['wc_whatsapp_webservice_nonce'] ) ) {
            check_admin_referer( 'wc_whatsapp_waappir_save_webservice', 'wc_whatsapp_webservice_nonce' );

            $enabled    = isset( $_POST['wc_whatsapp_enabled'] ) ? 'yes' : 'no';
            $auth_token = isset( $_POST['wc_whatsapp_auth_token'] ) ? sanitize_text_field( wp_unslash( $_POST['wc_whatsapp_auth_token'] ) ) : '';

            update_option( 'wc_whatsapp_waappir_enabled', $enabled );
            update_option( 'wc_whatsapp_waappir_api_url', 'https://panel.waapp.ir/api/v1/send' );
            update_option( 'wc_whatsapp_waappir_auth_token', $auth_token );

            add_settings_error( 'wc_whatsapp_messages', 'wc_whatsapp_waappir_webservice_saved', __( 'تنظیمات وب‌سرویس ذخیره شد.', 'wc-whatsapp-waappir' ), 'updated' );
        }


        // Handle admin message tab submission
        if ( 'admin_message' === $active_tab && isset( $_POST['wc_whatsapp_waappir_admin_msg_nonce'] ) ) {
            check_admin_referer( 'wc_whatsapp_waappir_save_admin_msg', 'wc_whatsapp_waappir_admin_msg_nonce' );

            $admin_enabled = isset( $_POST['wc_whatsapp_admin_enabled'] ) ? 'yes' : 'no';
            $admin_phone   = isset( $_POST['wc_whatsapp_admin_phone'] ) ? sanitize_text_field( wp_unslash( $_POST['wc_whatsapp_admin_phone'] ) ) : '';
            $selected_statuses = isset( $_POST['wc_whatsapp_admin_statuses'] ) ? array_map( 'sanitize_text_field', wp_unslash( $_POST['wc_whatsapp_admin_statuses'] ) ) : array();
            
            update_option( 'wc_whatsapp_waappir_admin_enabled', $admin_enabled );
            update_option( 'wc_whatsapp_waappir_admin_phone', $admin_phone );
            update_option( 'wc_whatsapp_waappir_admin_statuses', $selected_statuses );

            // Save status messages
            $order_statuses = wc_get_order_statuses();
            foreach ( $order_statuses as $status_key => $status_label ) {
                $clean_status = str_replace( 'wc-', '', $status_key );
                $msg_key = 'wc_whatsapp_waappir_admin_msg_' . $clean_status;
                $message = isset( $_POST[$msg_key] ) ? wp_kses_post( wp_unslash( $_POST[$msg_key] ) ) : '';
                update_option( $msg_key, $message );
            }

            // Save stock messages
            $low_stock_msg  = isset( $_POST['wc_whatsapp_admin_low_stock'] ) ? wp_kses_post( wp_unslash( $_POST['wc_whatsapp_admin_low_stock'] ) ) : '';
            $out_stock_msg = isset( $_POST['wc_whatsapp_admin_out_stock'] ) ? wp_kses_post( wp_unslash( $_POST['wc_whatsapp_admin_out_stock'] ) ) : '';
            
            update_option( 'wc_whatsapp_waappir_admin_low_stock', $low_stock_msg );
            update_option( 'wc_whatsapp_waappir_admin_out_stock', $out_stock_msg );

            add_settings_error( 'wc_whatsapp_messages', 'wc_whatsapp_waappir_admin_msg_saved', __( 'تنظیمات پیام به مدیر کل ذخیره شد.', 'wc-whatsapp-waappir' ), 'updated' );
        }

        // Handle customer message tab submission
        if ( 'customer_message' === $active_tab && isset( $_POST['wc_whatsapp_waappir_customer_msg_nonce'] ) ) {
            check_admin_referer( 'wc_whatsapp_waappir_save_customer_msg', 'wc_whatsapp_waappir_customer_msg_nonce' );

            $customer_enabled = isset( $_POST['wc_whatsapp_customer_enabled'] ) ? 'yes' : 'no';
            $customer_new_order_enabled = isset( $_POST['wc_whatsapp_customer_new_order_enabled'] ) ? 'yes' : 'no';
            $selected_statuses = isset( $_POST['wc_whatsapp_customer_statuses'] ) ? array_map( 'sanitize_text_field', wp_unslash( $_POST['wc_whatsapp_customer_statuses'] ) ) : array();

            update_option( 'wc_whatsapp_waappir_customer_enabled', $customer_enabled );
            update_option( 'wc_whatsapp_waappir_customer_new_order_enabled', $customer_new_order_enabled );
            update_option( 'wc_whatsapp_waappir_customer_statuses', $selected_statuses );

            // Save new order message
            $new_order_msg = isset( $_POST['wc_whatsapp_customer_new_order_msg'] ) ? wp_kses_post( wp_unslash( $_POST['wc_whatsapp_customer_new_order_msg'] ) ) : '';
            update_option( 'wc_whatsapp_waappir_customer_new_order_msg', $new_order_msg );

            // Save status messages
            $order_statuses = wc_get_order_statuses();
            foreach ( $order_statuses as $status_key => $status_label ) {
                $clean_status = str_replace( 'wc-', '', $status_key );
                $msg_key = 'wc_whatsapp_waappir_customer_msg_' . $clean_status;
                $message = isset( $_POST[$msg_key] ) ? wp_kses_post( wp_unslash( $_POST[$msg_key] ) ) : '';
                update_option( $msg_key, $message );
            }

            add_settings_error( 'wc_whatsapp_messages', 'wc_whatsapp_waappir_customer_msg_saved', __( 'تنظیمات پیام به مشتری ذخیره شد.', 'wc-whatsapp-waappir' ), 'updated' );
        }

        settings_errors( 'wc_whatsapp_messages' );
    }

    /**
     * Render Webservice tab.
     */
    protected static function render_webservice_tab() {
        $enabled    = get_option( 'wc_whatsapp_waappir_enabled', 'no' );
        $auth_token = get_option( 'wc_whatsapp_waappir_auth_token', '' );
        ?>
        <form method="post" action="">
            <?php wp_nonce_field( 'wc_whatsapp_waappir_save_webservice', 'wc_whatsapp_webservice_nonce' ); ?>

            <div style="background: #fff3cd; border: 1px solid #ffeaa7; padding: 20px; margin-bottom: 20px; border-radius: 5px;">
                <h3 style="margin-top: 0; color: #856404;"><?php esc_html_e( '⚠️ فعالسازی سرویس واتساپ', 'wc-whatsapp-waappir' ); ?></h3>
                <p style="margin-bottom: 15px; color: #856404;"><?php esc_html_e( 'برای فعالسازی و استفاده واقعی از سرویس ارسال پیام واتساپ، نیاز به خرید پلن اشتراکی دارید.', 'wc-whatsapp-waappir' ); ?></p>
                <p style="margin-bottom: 15px; color: #856404;"><?php esc_html_e( 'با خرید پلن مناسب، توکن API دریافت کرده و می‌توانید از تمام امکانات سرویس استفاده کنید.', 'wc-whatsapp-waappir' ); ?></p>
                <a href="https://panel.waapp.ir/plans" target="_blank" class="button button-primary" style="background: #007cba; border-color: #007cba; color: #fff; text-decoration: none; padding: 8px 16px; border-radius: 4px; display: inline-block;">
                    <?php esc_html_e( 'مشاهده پلن‌های اشتراکی', 'wc-whatsapp-waappir' ); ?>
                </a>
            </div>

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><?php esc_html_e( 'فعال سازی وب‌سرویس', 'wc-whatsapp-waappir' ); ?></th>
                    <td>
                        <label for="wc_whatsapp_enabled">
                            <input type="checkbox" id="wc_whatsapp_enabled" name="wc_whatsapp_enabled" value="yes" <?php checked( $enabled, 'yes' ); ?> />
                            <?php esc_html_e( 'فعال یا غیرفعال کردن اتصال به واتساپ.', 'wc-whatsapp-waappir' ); ?>
                        </label>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e( 'Authorization (توکن)', 'wc-whatsapp-waappir' ); ?></th>
                    <td>
                        <input type="password" class="regular-text ltr" id="wc_whatsapp_auth_token" name="wc_whatsapp_auth_token" value="<?php echo esc_attr( $auth_token ); ?>" />
                        <p class="description"><?php esc_html_e( 'توکن دسترسی برای هدر Authorization.', 'wc-whatsapp-waappir' ); ?></p>
                    </td>
                </tr>
            </table>

            <?php submit_button( __( 'ذخیره تغییرات', 'wc-whatsapp-waappir' ) ); ?>
        </form>
        <?php
    }


    /**
     * Render Admin Message tab.
     */
    protected static function render_admin_message_tab() {
        $enabled    = get_option( 'wc_whatsapp_waappir_admin_enabled', 'no' );
        $admin_phone = get_option( 'wc_whatsapp_waappir_admin_phone', '' );
        $selected_statuses = get_option( 'wc_whatsapp_waappir_admin_statuses', array() );
        $order_statuses = wc_get_order_statuses();
        ?>
        <form method="post" action="">
            <?php wp_nonce_field( 'wc_whatsapp_waappir_save_admin_msg', 'wc_whatsapp_waappir_admin_msg_nonce' ); ?>

            <h2><?php esc_html_e( 'تنظیمات ارسال پیام به مدیر کل', 'wc-whatsapp-waappir' ); ?></h2>
            
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><?php esc_html_e( 'فعال سازی ارسال به مدیر', 'wc-whatsapp-waappir' ); ?></th>
                    <td>
                        <label for="wc_whatsapp_admin_enabled">
                            <input type="checkbox" id="wc_whatsapp_admin_enabled" name="wc_whatsapp_admin_enabled" value="yes" <?php checked( $enabled, 'yes' ); ?> />
                            <?php esc_html_e( 'فعال یا غیرفعال کردن ارسال پیام به مدیر کل.', 'wc-whatsapp-waappir' ); ?>
                        </label>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e( 'شماره واتساپ مدیر', 'wc-whatsapp-waappir' ); ?></th>
                    <td>
                        <input type="text" class="regular-text ltr" id="wc_whatsapp_admin_phone" name="wc_whatsapp_admin_phone" value="<?php echo esc_attr( $admin_phone ); ?>" placeholder="+989123456789" />
                        <p class="description"><?php esc_html_e( 'شماره واتساپ مدیر کل برای دریافت اطلاع‌رسانی‌ها.', 'wc-whatsapp-waappir' ); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e( 'وضعیت‌های دریافت پیام', 'wc-whatsapp-waappir' ); ?></th>
                    <td>
                        <fieldset>
                            <?php foreach ( $order_statuses as $status_key => $status_label ) : 
                                $clean_status = str_replace( 'wc-', '', $status_key );
                            ?>
                                <label style="display: block; margin-bottom: 5px;">
                                    <input type="checkbox" name="wc_whatsapp_admin_statuses[]" value="<?php echo esc_attr( $clean_status ); ?>" <?php checked( in_array( $clean_status, $selected_statuses, true ) ); ?> />
                                    <?php echo esc_html( $status_label ); ?>
                                </label>
                            <?php endforeach; ?>
                        </fieldset>
                        <p class="description"><?php esc_html_e( 'انتخاب کنید در کدام وضعیت‌های سفارش، پیام به مدیر ارسال شود.', 'wc-whatsapp-waappir' ); ?></p>
                    </td>
                </tr>
            </table>

            <h3><?php esc_html_e( 'پیام‌های وضعیت سفارش', 'wc-whatsapp-waappir' ); ?></h3>
            <table class="form-table" role="presentation">
                <?php foreach ( $order_statuses as $status_key => $status_label ) :
                    $clean_status = str_replace( 'wc-', '', $status_key );
                    $msg_key = 'wc_whatsapp_waappir_admin_msg_' . $clean_status;

                    // Default messages for admin based on order status
                    $default_messages = array(
                        'pending'    => 'سفارش جدید با شماره {order_id} دریافت شد. مبلغ: {price} تومان. منتظر پرداخت مشتری هستیم.' . "\n\n" . '[این پیام به صورت خودکار ارسال شده است]',
                        'processing' => 'سفارش {order_id} در حال پردازش است. مبلغ: {price} تومان.' . "\n\n" . '[این پیام به صورت خودکار ارسال شده است]',
                        'on-hold'    => 'سفارش {order_id} در وضعیت انتظار قرار گرفت. مبلغ: {price} تومان.' . "\n\n" . '[این پیام به صورت خودکار ارسال شده است]',
                        'completed'  => 'سفارش {order_id} تکمیل شد. مبلغ: {price} تومان. محصولات: {all_items}' . "\n\n" . '[این پیام به صورت خودکار ارسال شده است]',
                        'cancelled'  => 'سفارش {order_id} لغو شد. مبلغ: {price} تومان.' . "\n\n" . '[این پیام به صورت خودکار ارسال شده است]',
                        'refunded'   => 'سفارش {order_id} بازپرداخت شد. مبلغ: {price} تومان.' . "\n\n" . '[این پیام به صورت خودکار ارسال شده است]',
                        'failed'     => 'پرداخت سفارش {order_id} ناموفق بود. مبلغ: {price} تومان.' . "\n\n" . '[این پیام به صورت خودکار ارسال شده است]',
                    );

                    $default_message = isset($default_messages[$clean_status]) ? $default_messages[$clean_status] : '';
                    $message = get_option( $msg_key, $default_message );
                ?>
                    <tr>
                        <th scope="row"><?php echo esc_html( $status_label ); ?></th>
                        <td>
                            <textarea name="<?php echo esc_attr( $msg_key ); ?>" rows="3" class="large-text"><?php echo esc_textarea( $message ); ?></textarea>
                            <p class="description"><?php esc_html_e( 'پیامی که هنگام تغییر به این وضعیت ارسال می‌شود.', 'wc-whatsapp-waappir' ); ?></p>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </table>

            <h3><?php esc_html_e( 'پیام‌های موجودی انبار', 'wc-whatsapp-waappir' ); ?></h3>
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><?php esc_html_e( 'پیام کم بودن موجودی', 'wc-whatsapp-waappir' ); ?></th>
                    <td>
                        <textarea name="wc_whatsapp_admin_low_stock" rows="3" class="large-text"><?php echo esc_textarea( get_option( 'wc_whatsapp_waappir_admin_low_stock', '⚠️ موجودی محصول {product_name} (کد: {product_sku}) کم است. موجودی فعلی: {stock_qty} عدد. لطفاً موجودی را بررسی و تکمیل کنید.' . "\n\n" . '[این پیام به صورت خودکار ارسال شده است]' ) ); ?></textarea>
                        <p class="description"><?php esc_html_e( 'پیام هنگام کم شدن موجودی محصول. شورت کدها: {product_name}, {product_id}, {product_sku}, {stock_qty}', 'wc-whatsapp-waappir' ); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e( 'پیام تمام شدن موجودی', 'wc-whatsapp-waappir' ); ?></th>
                    <td>
                        <textarea name="wc_whatsapp_admin_out_stock" rows="3" class="large-text"><?php echo esc_textarea( get_option( 'wc_whatsapp_waappir_admin_out_stock', '🚨 موجودی محصول {product_name} (کد: {product_sku}) تمام شده است! لطفاً موجودی را بررسی و محصول را مجدداً تأمین کنید.' . "\n\n" . '[این پیام به صورت خودکار ارسال شده است]' ) ); ?></textarea>
                        <p class="description"><?php esc_html_e( 'پیام هنگام تمام شدن موجودی محصول. شورت کدها: {product_name}, {product_id}, {product_sku}', 'wc-whatsapp-waappir' ); ?></p>
                    </td>
                </tr>
            </table>

            <?php self::render_shortcode_guide(); ?>

            <?php submit_button( __( 'ذخیره تغییرات', 'wc-whatsapp-waappir' ) ); ?>
        </form>
        <?php
    }

    /**
     * Render Customer Message tab.
     */
    protected static function render_customer_message_tab() {
        $enabled    = get_option( 'wc_whatsapp_waappir_customer_enabled', 'no' );
        $new_order_enabled = get_option( 'wc_whatsapp_waappir_customer_new_order_enabled', 'no' );
        $selected_statuses = get_option( 'wc_whatsapp_waappir_customer_statuses', array() );
        $order_statuses = wc_get_order_statuses();
        ?>
        <form method="post" action="">
            <?php wp_nonce_field( 'wc_whatsapp_waappir_save_customer_msg', 'wc_whatsapp_waappir_customer_msg_nonce' ); ?>

            <h2><?php esc_html_e( 'تنظیمات ارسال پیام به مشتری', 'wc-whatsapp-waappir' ); ?></h2>

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><?php esc_html_e( 'فعال سازی ارسال به مشتری', 'wc-whatsapp-waappir' ); ?></th>
                    <td>
                        <label for="wc_whatsapp_customer_enabled">
                            <input type="checkbox" id="wc_whatsapp_customer_enabled" name="wc_whatsapp_customer_enabled" value="yes" <?php checked( $enabled, 'yes' ); ?> />
                            <?php esc_html_e( 'فعال یا غیرفعال کردن ارسال پیام به مشتری.', 'wc-whatsapp-waappir' ); ?>
                        </label>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e( 'ارسال پیام ثبت سفارش', 'wc-whatsapp-waappir' ); ?></th>
                    <td>
                        <label for="wc_whatsapp_customer_new_order_enabled">
                            <input type="checkbox" id="wc_whatsapp_customer_new_order_enabled" name="wc_whatsapp_customer_new_order_enabled" value="yes" <?php checked( $new_order_enabled, 'yes' ); ?> />
                            <?php esc_html_e( 'ارسال پیام خودکار هنگام ثبت سفارش جدید.', 'wc-whatsapp-waappir' ); ?>
                        </label>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e( 'وضعیت‌های دریافت پیام', 'wc-whatsapp-waappir' ); ?></th>
                    <td>
                        <fieldset>
                            <?php foreach ( $order_statuses as $status_key => $status_label ) :
                                $clean_status = str_replace( 'wc-', '', $status_key );
                            ?>
                                <label style="display: block; margin-bottom: 5px;">
                                    <input type="checkbox" name="wc_whatsapp_customer_statuses[]" value="<?php echo esc_attr( $clean_status ); ?>" <?php checked( in_array( $clean_status, $selected_statuses, true ) ); ?> />
                                    <?php echo esc_html( $status_label ); ?>
                                </label>
                            <?php endforeach; ?>
                        </fieldset>
                        <p class="description"><?php esc_html_e( 'انتخاب کنید در کدام وضعیت‌های سفارش، پیام به مشتری ارسال شود.', 'wc-whatsapp-waappir' ); ?></p>
                    </td>
                </tr>
            </table>

            <h3><?php esc_html_e( 'پیام ثبت سفارش', 'wc-whatsapp-waappir' ); ?></h3>
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><?php esc_html_e( 'پیام ثبت سفارش', 'wc-whatsapp-waappir' ); ?></th>
                    <td>
                        <textarea name="wc_whatsapp_customer_new_order_msg" rows="3" class="large-text"><?php echo esc_textarea( get_option( 'wc_whatsapp_waappir_customer_new_order_msg', '✅ سفارش شما با شماره {order_id} ثبت شد. مبلغ کل: {price} تومان. به زودی با شما تماس خواهیم گرفت. از خرید شما متشکر هستیم! 🛒' . "\n\n" . '[این پیام به صورت خودکار ارسال شده است]' ) ); ?></textarea>
                        <p class="description"><?php esc_html_e( 'پیامی که هنگام ثبت سفارش جدید برای مشتری ارسال می‌شود.', 'wc-whatsapp-waappir' ); ?></p>
                    </td>
                </tr>
            </table>

            <h3><?php esc_html_e( 'پیام‌های وضعیت سفارش', 'wc-whatsapp-waappir' ); ?></h3>
            <table class="form-table" role="presentation">
                <?php foreach ( $order_statuses as $status_key => $status_label ) :
                    $clean_status = str_replace( 'wc-', '', $status_key );
                    $msg_key = 'wc_whatsapp_waappir_customer_msg_' . $clean_status;

                    // Default messages for customer based on order status
                    $default_messages = array(
                        'pending'    => 'سفارش شما با شماره {order_id} دریافت شد. لطفاً برای تکمیل خرید، پرداخت را انجام دهید. مبلغ: {price} تومان' . "\n\n" . '[این پیام به صورت خودکار ارسال شده است]',
                        'processing' => 'سفارش شما با شماره {order_id} در حال آماده‌سازی است. به زودی با شما تماس خواهیم گرفت.' . "\n\n" . '[این پیام به صورت خودکار ارسال شده است]',
                        'on-hold'    => 'سفارش شما با شماره {order_id} در انتظار تأیید نهایی است. با شما تماس خواهیم گرفت.' . "\n\n" . '[این پیام به صورت خودکار ارسال شده است]',
                        'completed'  => 'سفارش شما با شماره {order_id} تکمیل شد و آماده ارسال است. از خرید شما متشکر هستیم! ❤️' . "\n\n" . '[این پیام به صورت خودکار ارسال شده است]',
                        'cancelled'  => 'سفارش شما با شماره {order_id} لغو شد. در صورت سوال با ما تماس بگیرید.' . "\n\n" . '[این پیام به صورت خودکار ارسال شده است]',
                        'refunded'   => 'مبلغ سفارش {order_id} به حساب شما بازپرداخت شد. از خرید شما متشکر هستیم.' . "\n\n" . '[این پیام به صورت خودکار ارسال شده است]',
                        'failed'     => 'پرداخت سفارش {order_id} ناموفق بود. لطفاً مجدداً تلاش کنید یا با پشتیبانی تماس بگیرید.' . "\n\n" . '[این پیام به صورت خودکار ارسال شده است]',
                    );

                    $default_message = isset($default_messages[$clean_status]) ? $default_messages[$clean_status] : '';
                    $message = get_option( $msg_key, $default_message );
                ?>
                    <tr>
                        <th scope="row"><?php echo esc_html( $status_label ); ?></th>
                        <td>
                            <textarea name="<?php echo esc_attr( $msg_key ); ?>" rows="3" class="large-text"><?php echo esc_textarea( $message ); ?></textarea>
                            <p class="description"><?php esc_html_e( 'پیامی که هنگام تغییر به این وضعیت برای مشتری ارسال می‌شود.', 'wc-whatsapp-waappir' ); ?></p>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </table>

            <?php self::render_shortcode_guide(); ?>

            <?php submit_button( __( 'ذخیره تغییرات', 'wc-whatsapp-waappir' ) ); ?>
        </form>
        <?php
    }

    /**
     * Render shortcode guide.
     */
    protected static function render_shortcode_guide() {
        ?>
        <div style="background: #f5f5f5; border: 1px solid #ddd; padding: 15px; margin-top: 20px;">
            <h3><?php esc_html_e( 'راهنمای شورت کدهای قابل استفاده', 'wc-whatsapp-waappir' ); ?></h3>
            
            <h4><?php esc_html_e( 'جزئیات سفارش:', 'wc-whatsapp-waappir' ); ?></h4>
            <p style="font-family: monospace; line-height: 1.8;">
                {mobile} = شماره موبایل مشتری | {phone} = شماره تلفن مشتری | {email} = ایمیل مشتری | {status} = وضعیت سفارش<br>
                {all_items} = محصولات سفارش | {all_items_full} = محصولات با نام کامل متغیر | {all_items_qty} = محصولات بهمراه تعداد<br>
                {count_items} = تعداد محصولات | {price} = مبلغ سفارش | {post_id} = شماره سفارش اصلی | {order_id} = شماره سفارش<br>
                {transaction_id} = شماره تراکنش | {date} = تاریخ سفارش | {description} = توضیحات مشتری<br>
                {payment_method} = روش پرداخت | {shipping_method} = روش ارسال
            </p>

            <h4><?php esc_html_e( 'جزئیات صورت حساب:', 'wc-whatsapp-waappir' ); ?></h4>
            <p style="font-family: monospace; line-height: 1.8;">
                {b_first_name} = نام | {b_last_name} = نام خانوادگی | {b_company} = شرکت | {b_country} = کشور<br>
                {b_state} = استان | {b_city} = شهر | {b_address_1} = آدرس 1 | {b_address_2} = آدرس 2 | {b_postcode} = کد پستی
            </p>

            <h4><?php esc_html_e( 'جزئیات حمل و نقل:', 'wc-whatsapp-waappir' ); ?></h4>
            <p style="font-family: monospace; line-height: 1.8;">
                {sh_first_name} = نام | {sh_last_name} = نام خانوادگی | {sh_company} = شرکت | {sh_country} = کشور<br>
                {sh_state} = استان | {sh_city} = شهر | {sh_address_1} = آدرس 1 | {sh_address_2} = آدرس 2 | {sh_postcode} = کد پستی
            </p>

            <h4><?php esc_html_e( 'اطلاعات رهگیری:', 'wc-whatsapp-waappir' ); ?></h4>
            <p style="font-family: monospace; line-height: 1.8;">
                {post_tracking_code} = کد رهگیری پستی | {post_tracking_url} = آدرس اینترنتی رهگیری پستی
            </p>
        </div>
        <?php
    }

    /**
     * Render Features tab.
     */
    protected static function render_features_tab() {
        ?>
        <div class="wc-whatsapp-features">
            <h2><?php esc_html_e( 'ویژگی‌های قدرتمند', 'wc-whatsapp-waappir' ); ?></h2>
            <p><?php esc_html_e( 'همه چیزهایی که برای مدیریت واتساپ نیاز دارید در یک پنل', 'wc-whatsapp-waappir' ); ?></p>

            <div class="wc-whatsapp-feature-grid">
                <div class="wc-whatsapp-feature-item">
                    <h3><?php esc_html_e( 'ارسال پیام از طریق API', 'wc-whatsapp-waappir' ); ?></h3>
                    <p><?php esc_html_e( 'ارسال پیام‌های متنی و رسانه از طریق REST API با استفاده از توکن‌های امنیتی. پشتیبانی کامل از تمام انواع پیام‌های واتساپ.', 'wc-whatsapp-waappir' ); ?></p>
                </div>

                <div class="wc-whatsapp-feature-item">
                    <h3><?php esc_html_e( 'مدیریت چند حساب', 'wc-whatsapp-waappir' ); ?></h3>
                    <p><?php esc_html_e( 'مدیریت و استفاده از چندین حساب واتساپ به صورت همزمان در یک پنل واحد. اتصال و مدیریت آسان تمام حساب‌ها.', 'wc-whatsapp-waappir' ); ?></p>
                </div>

                <div class="wc-whatsapp-feature-item">
                    <h3><?php esc_html_e( 'API Token امن', 'wc-whatsapp-waappir' ); ?></h3>
                    <p><?php esc_html_e( 'استفاده از توکن‌های API برای دسترسی امن به سرویس. امکان محدودسازی IP و Rate Limiting برای امنیت بیشتر.', 'wc-whatsapp-waappir' ); ?></p>
                </div>

                <div class="wc-whatsapp-feature-item">
                    <h3><?php esc_html_e( 'صف پیام‌ها', 'wc-whatsapp-waappir' ); ?></h3>
                    <p><?php esc_html_e( 'پیام‌ها در صف قرار می‌گیرند و به صورت خودکار ارسال می‌شوند حتی اگر حساب متصل نباشد. مدیریت هوشمند صف.', 'wc-whatsapp-waappir' ); ?></p>
                </div>

                <div class="wc-whatsapp-feature-item">
                    <h3><?php esc_html_e( 'Webhook پشتیبانی', 'wc-whatsapp-waappir' ); ?></h3>
                    <p><?php esc_html_e( 'دریافت خودکار پیام‌های دریافتی از طریق Webhook. اتصال آسان به سیستم‌های دیگر و اتوماسیون کامل.', 'wc-whatsapp-waappir' ); ?></p>
                </div>

                <div class="wc-whatsapp-feature-item">
                    <h3><?php esc_html_e( 'افزونه‌های آماده', 'wc-whatsapp-waappir' ); ?></h3>
                    <p><?php esc_html_e( 'افزونه‌های آماده برای سیستم‌های محبوب مانند وردپرس، جوملا و غیره. اتصال آسان به CRM، فروشگاه‌های آنلاین و پنل‌های مدیریتی.', 'wc-whatsapp-waappir' ); ?></p>
                </div>

                <div class="wc-whatsapp-feature-item">
                    <h3><?php esc_html_e( 'ایجاد اتوماتیک لیست مخاطبین', 'wc-whatsapp-waappir' ); ?></h3>
                    <p><?php esc_html_e( 'ایجاد خودکار لیست مخاطبین از طریق سرویس‌های مختلف. همگام‌سازی با سیستم‌های CRM و بانک‌های اطلاعاتی موجود.', 'wc-whatsapp-waappir' ); ?></p>
                </div>

                <div class="wc-whatsapp-feature-item">
                    <h3><?php esc_html_e( 'ایمپورت مخاطبین از اکسل', 'wc-whatsapp-waappir' ); ?></h3>
                    <p><?php esc_html_e( 'آپلود و ایمپورت مخاطبین از فایل‌های Excel. دسته‌بندی هوشمند مخاطبین و مدیریت گروهی برای ارسال پیام‌های هدفمند.', 'wc-whatsapp-waappir' ); ?></p>
                </div>

                <div class="wc-whatsapp-feature-item">
                    <h3><?php esc_html_e( 'پلن‌های اقتصادی و جامع', 'wc-whatsapp-waappir' ); ?></h3>
                    <p><?php esc_html_e( 'پلن‌های ارزان قیمت با پوشش کامل. بدون محدودیت در تعداد پیام، کاراکتر، فایل، عکس و ویدیو. پرداخت یکباره و استفاده نامحدود.', 'wc-whatsapp-waappir' ); ?></p>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render About tab.
     */
    protected static function render_about_tab() {
        ?>
        <div class="wc-whatsapp-about">
            <h2><?php esc_html_e( 'درباره ما', 'wc-whatsapp-waappir' ); ?></h2>
            <p><?php esc_html_e( 'ما یک پلتفرم مدیریت واتساپ مبتنی بر API هستیم که به کسب‌وکارها کمک می‌کنیم پیام‌ها را سریع‌تر، امن‌تر و هوشمندانه‌تر مدیریت کنند.', 'wc-whatsapp-waappir' ); ?></p>

            <h3><?php esc_html_e( 'ماموریت ما', 'wc-whatsapp-waappir' ); ?></h3>
            <p><?php esc_html_e( 'ایجاد دسترسی ساده و پایدار به واتساپ برای توسعه‌دهندگان و تیم‌های پشتیبانی.', 'wc-whatsapp-waappir' ); ?></p>

            <h3><?php esc_html_e( 'چشم‌انداز', 'wc-whatsapp-waappir' ); ?></h3>
            <p><?php esc_html_e( 'ساخت زیرساخت پیام‌رسانی قابل اعتماد برای مقیاس‌های بالا با تمرکز بر امنیت و سرعت.', 'wc-whatsapp-waappir' ); ?></p>

            <h3><?php esc_html_e( 'ارزش‌ها', 'wc-whatsapp-waappir' ); ?></h3>
            <p><?php esc_html_e( 'امنیت داده، شفافیت، پشتیبانی سریع و تجربه کاربری ساده برای همه مشتریان.', 'wc-whatsapp-waappir' ); ?></p>

            <p><?php esc_html_e( 'به ما بپیوندید تا ارتباطات واتساپی کسب‌وکار شما همیشه پایدار و قابل اعتماد باشد.', 'wc-whatsapp-waappir' ); ?></p>

            <p>
                <a href="https://panel.waapp.ir" target="_blank" class="button button-secondary"><?php esc_html_e( 'دیدن خانهٔ افزونه', 'wc-whatsapp-waappir' ); ?></a>
                <a href="https://panel.waapp.ir/plans" target="_blank" class="button button-primary"><?php esc_html_e( 'مشاهده پلن‌ها', 'wc-whatsapp-waappir' ); ?></a>
            </p>
        </div>
        <?php
    }
}
